/*
 * can_rx_task
 *
 * Copyright (C) 2022 Texas Instruments Incorporated
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/******************************************************************************
 *
 * vCANTask is called by main() to configure the CAN0 module for receiving data
 * at a 500kHz bit rate and enables interrupts.  It then creates the task which
 * will handle the CAN data processing.
 *
 * prvReceiveTask1 processes all the data received on different message
 * objects.  The concept of event group is used in this example where multiple
 * events (e.g. notifications from different interrupt sources) can unblock a
 * BLOCKed task.  When a message object receives data, it interrupts the
 * processor.  In the CANIntHandler, xTaskNotifyFromISR is called and passed
 * with an unique notification value associated with the message object.  In
 * the prvReceiveTask1 task, it is unblocked when any one of the recognized
 * notification values is received. Depending on which notification value is
 * received, the associated message object is read and processed.
 *
 * This example uses the following peripherals and I/O signals on the
 * EK-TM4C123GXL:
 * - CAN0RX - PE4
 * - CAN0TX - PE5
 *
 */

/* Standard includes. */
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <stdbool.h>

/* Kernel includes. */
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"

/* Hardware includes. */
#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "inc/hw_can.h"
#include "driverlib/gpio.h"
#include "driverlib/interrupt.h"
#include "driverlib/can.h"
#include "driverlib/pin_map.h"
#include "driverlib/rom.h"
#include "driverlib/rom_map.h"
#include "driverlib/sysctl.h"
#include "drivers/rtos_hw_drivers.h"
#include "utils/uartstdio.h"
#include "uart_task.h"
/*-----------------------------------------------------------*/

/*
 * These define the interrupt source for each CAN Message Object.
 */
#define MSGOBJ1 0x1
#define MSGOBJ2 0x2
#define MSGOBJ3 0x4

/*
 * A counter that keeps track of the number of times the TX interrupt has
 * occurred, which should match the number of TX messages that were sent.
 */
volatile uint32_t g_ui32MsgCount = 0;

/*
 * A flag to indicate that some transmission error occurred.
 */
volatile bool g_bErrFlag = 0;

/*
 * The notification used by the task.
 */
TaskHandle_t xTaskMsgObjHandle = NULL;

/*
 * The tasks as described in the comments at the top of this file.
 */
static void prvReceiveTask1( void *pvParameters );

/*
 * This function prints some information about the CAN message to the
 * serial port for information purposes only.
 */
void PrintCANMessageInfo(tCANMsgObject *psCANMsg, uint32_t ui32MsgObj);

/*
 * Called by main() to create the CAN application.
 */
void vCANTask( void );

/*
 * Hardware configuration for the CAN peripheral.
 */
static void prvConfigureCAN( void );
/*-----------------------------------------------------------*/

void vCANTask( void )
{
    /* Configure the CAN 0 peripheral. */
    prvConfigureCAN();

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name RX by value processing task - for debug only as it is
     *    not used by the kernel.
     *  - The size of the stack to allocate to the task.
     *  - No parameter passed to the task
     *  - The priority assigned to the task.
     *  - The task handle is xTaskMsgObjHandle */
    xTaskCreate( prvReceiveTask1,
                 "RX by value",
                 configMINIMAL_STACK_SIZE,
                 NULL,
                 tskIDLE_PRIORITY + 1,
                 &xTaskMsgObjHandle );
}
/*-----------------------------------------------------------*/

static void prvReceiveTask1( void *pvParameters )
{
BaseType_t xResult;
uint32_t ulNotifiedValue;
tCANMsgObject sCANMessage;
uint8_t pui8MsgData[8];

    /* Initialize a message object to receive CAN messages with ID 0x1001.
     * The expected ID must be set along with the mask to indicate that all
     * bits in the ID must match. */
    sCANMessage.ui32MsgID = 0x1001;
    sCANMessage.ui32MsgIDMask = 0xfffff;
    sCANMessage.ui32Flags = (MSG_OBJ_RX_INT_ENABLE | MSG_OBJ_USE_ID_FILTER |
            MSG_OBJ_EXTENDED_ID);
    sCANMessage.ui32MsgLen = 8;
    sCANMessage.pui8MsgData = pui8MsgData;

    /* Now load the message object into the CAN peripheral message object 1.
     * Once loaded the CAN will receive any messages with this CAN ID into
     * this message object, and an interrupt will occur. */
    CANMessageSet(CAN0_BASE, 1, &sCANMessage, MSG_OBJ_TYPE_RX);

    /* Change the ID to 0x2001, and load into message object 2 which will be
     * used for receiving any CAN messages with this ID.  Since only the CAN
     * ID field changes, we don't need to reload all the other fields. */
    sCANMessage.ui32MsgID = 0x2001;
    CANMessageSet(CAN0_BASE, 2, &sCANMessage, MSG_OBJ_TYPE_RX);

    /* Change the ID to 0x3001, and load into message object 3 which will be
     * used for receiving any CAN messages with this ID.  Since the filtering
     * mask is 0xFFFFF, it means that only Message ID 0x3001 originates
     * from can_multi_tx example will be accepted by Message Object and
     * Message ID 0x3002 from the transmitter will be filtered out. */
    sCANMessage.ui32MsgID = 0x3001;
    CANMessageSet(CAN0_BASE, 3, &sCANMessage, MSG_OBJ_TYPE_RX);

    prvUARTPrintf("CAN Multi-RX example\n");

    for (;;)
    {
        /* Wait to receive a notification sent directly to this task from the
         * interrupt service routine.
         * The xTaskNotifyWait parameters in order are:
         *  - Don't clear bits on entry.
         *  - Clear all bits on exit.
         *  - Storage location for the notified value.
         *  - How many RTOS ticks to wait for a notification. */
        xResult = xTaskNotifyWait( pdFALSE,
                                   MSGOBJ1 | MSGOBJ2 | MSGOBJ3,
                                   &ulNotifiedValue,
                                   portMAX_DELAY );

        if( xResult == pdPASS )
        {
            /* A notification was received.  See which bits were set. */
            if( ( ulNotifiedValue & MSGOBJ1 ) == MSGOBJ1 )
            {
                /* Read the message from the CAN Msg Obj 1. */
                CANMessageGet(CAN0_BASE, 1, &sCANMessage, 0);

                /* Display the received msg from MsgObj 1. */
                PrintCANMessageInfo(&sCANMessage, 1);
            }

            if( ( ulNotifiedValue & MSGOBJ2 ) == MSGOBJ2 )
            {
                /* Read the message from the CAN Msg Obj 2. */
                CANMessageGet(CAN0_BASE, 2, &sCANMessage, 0);

                /* Display the received msg from MsgObj 2. */
                PrintCANMessageInfo(&sCANMessage, 2);
            }

            if( ( ulNotifiedValue & MSGOBJ3 ) == MSGOBJ3 )
            {
                /* Read the message from the CAN Msg Obj 3. */
                CANMessageGet(CAN0_BASE, 3, &sCANMessage, 0);

                /* Display the received msg from MsgObj 3. */
                PrintCANMessageInfo(&sCANMessage, 3);
            }
        }
        else
        {
            /* Did not receive a notification within the expected time. */
        }
    }
}
/*-----------------------------------------------------------*/

static void prvConfigureCAN( void )
{
    /* Configure the GPIO pin muxing to select CAN0 functions for these pins.
     * This step selects which alternate function is available for these pins.
     * This is necessary if your part supports GPIO pin function muxing.
     * Consult the data sheet to see which functions are allocated per pin.
     * TODO: change this to select the port/pin you are using. */
    GPIOPinConfigure(GPIO_PE4_CAN0RX);
    GPIOPinConfigure(GPIO_PE5_CAN0TX);

    /* Enable the alternate function on the GPIO pins.  The above step selects
     * which alternate function is available.  This step actually enables the
     * alternate function instead of GPIO for these pins.
     * TODO: change this to match the port/pin you are using. */
    GPIOPinTypeCAN(GPIO_PORTE_BASE, GPIO_PIN_4 | GPIO_PIN_5);

    /* The GPIO port and pins have been set up for CAN.  The CAN peripheral
     * must be enabled. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_CAN0);

    /* Initialize the CAN controller. */
    CANInit(CAN0_BASE);

    /* Set up the bit rate for the CAN bus.  This function sets up the CAN
     * bus timing for a nominal configuration.  You can achieve more control
     * over the CAN bus timing by using the function CANBitTimingSet() instead
     * of this one, if needed. In this example, the CAN bus is set to 500 kHz.
     */
    CANBitRateSet(CAN0_BASE, configCPU_CLOCK_HZ, 500000);

    /* Enable interrupts on the CAN peripheral.  This example uses static
     * allocation of interrupt handlers which means the name of the handler
     * is in the vector table of startup code.  If you want to use dynamic
     * allocation of the vector table, then you must also call CANIntRegister()
     * here. */
    CANIntEnable(CAN0_BASE, CAN_INT_MASTER | CAN_INT_ERROR | CAN_INT_STATUS);

    /* Enable the CAN interrupt on the processor (NVIC). */
    IntEnable(INT_CAN0);

    /* Enable the CAN for operation. */
    CANEnable(CAN0_BASE);
}
/*-----------------------------------------------------------*/

void
PrintCANMessageInfo(tCANMsgObject *psCANMsg, uint32_t ui32MsgObj)
{
    unsigned int uIdx;

    /* Check to see if there is an indication that some messages were lost. */
    if(psCANMsg->ui32Flags & MSG_OBJ_DATA_LOST)
    {
        prvUARTPrintf("CAN message loss detected on message object %d\n",
                   ui32MsgObj);
    }

    /* Print out the contents of the message that was received. */
    prvUARTPrintf("Msg Obj=%u ID=0x%05X len=%u data=0x", ui32MsgObj,
               psCANMsg->ui32MsgID, psCANMsg->ui32MsgLen);
    for(uIdx = 0; uIdx < psCANMsg->ui32MsgLen; uIdx++)
    {
        prvUARTPrintf("%02X ", psCANMsg->pui8MsgData[uIdx]);
    }
    prvUARTPrintf("\n");
}
/*-----------------------------------------------------------*/

void
xCANHandler(void)
{
    uint32_t ui32Status;
    BaseType_t xTaskWoken = pdFALSE;


    /* Read the CAN interrupt status to find the cause of the interrupt. */
    ui32Status = CANIntStatus(CAN0_BASE, CAN_INT_STS_CAUSE);

    /* If the cause is a controller status interrupt, then get the status. */
    if(ui32Status == CAN_INT_INTID_STATUS)
    {
        /* Read the controller status.  This will return a field of status
         * error bits that can indicate various errors.  Error processing
         * is not done in this example for simplicity.  Refer to the
         * API documentation for details about the error status bits.
         * The act of reading this status will clear the interrupt. */
        ui32Status = CANStatusGet(CAN0_BASE, CAN_STS_CONTROL);

        /* Set a flag to indicate some errors may have occurred. */
        g_bErrFlag = 1;
    }

    /* Check if the cause is message object 1. */
    else if(ui32Status == 1)
    {
        /* Getting to this point means that the RX interrupt occurred on
         * message object 1, and the message reception is complete.  Clear the
         * message object interrupt. */
        CANIntClear(CAN0_BASE, 1);

        /* Increment a counter to keep track of how many messages have been
         * received.  In a real application this could be used to set flags to
         * indicate when a message is received. */
        g_ui32MsgCount++;

        /* Notify task to process the Msg Obj 1 data. */
        xTaskNotifyFromISR(xTaskMsgObjHandle,
                           MSGOBJ1,
                           eSetBits,
                           &xTaskWoken);

        /* Since a message was received, clear any error flags. */
        g_bErrFlag = 0;
    }

    /* Check if the cause is message object 2. */
    else if(ui32Status == 2)
    {
        CANIntClear(CAN0_BASE, 2);
        g_ui32MsgCount++;
        /* Notify task to process the Msg Obj 2 data. */
        xTaskNotifyFromISR(xTaskMsgObjHandle,
                           MSGOBJ2,
                           eSetBits,
                           &xTaskWoken);

        g_bErrFlag = 0;
    }

    /* Check if the cause is message object 3. */
    else if(ui32Status == 3)
    {
        CANIntClear(CAN0_BASE, 3);
        g_ui32MsgCount++;
        /* Notify task to process the Msg Obj 3 data. */
        xTaskNotifyFromISR(xTaskMsgObjHandle,
                           MSGOBJ3,
                           eSetBits,
                           &xTaskWoken);

        g_bErrFlag = 0;
    }

    /* Otherwise, something unexpected caused the interrupt.  This should
     * never happen. */
    else
    {
        /* Spurious interrupt handling can go here. */
    }

    /* This FreeRTOS API call will handle the context switch if it is
     * required or have no effect if that is not needed. */
    portYIELD_FROM_ISR( xTaskWoken );
}
/*-----------------------------------------------------------*/

void vApplicationTickHook( void )
{
    /* This function will be called by each tick interrupt if
        configUSE_TICK_HOOK is set to 1 in FreeRTOSConfig.h.  User code can be
        added here, but the tick hook is called from an interrupt context, so
        code must not attempt to block, and only the interrupt safe FreeRTOS API
        functions can be used (those that end in FromISR()). */

    /* Only the full demo uses the tick hook so there is no code is
        executed here. */
}

